/*
 * Decompiled with CFR 0.152.
 */
package com.mestrelab.license;

import com.mestrelab.license.exception.LicenseException;
import com.mestrelab.license.model.TLicense;
import com.mestrelab.license.model.TLicenseHolder;
import com.mestrelab.license.model.TProductError;
import com.mestrelab.license.util.Constants;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.spec.RSAPublicKeySpec;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.UUID;
import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TLicenseValidator {
    private List<TLicenseHolder> licenses = new ArrayList<TLicenseHolder>();
    private PublicKey publicKey;
    private SimpleDateFormat timeFormat = (SimpleDateFormat)DateFormat.getDateTimeInstance();
    private String licensePath;
    private String publicKeyPath;
    private Date fLastRun;
    private Date f1stRun;
    private Constants.TLicenseError fLastErr;
    private List<TProductError> fErrorsList = new ArrayList<TProductError>();
    private String fTempFile = System.getProperty("java.io.tmpdir") + System.getProperty("file.separator") + "runtimeLic.cache";
    private static final Logger logger = LoggerFactory.getLogger(TLicenseValidator.class);
    private static boolean hostLoaded = false;

    public TLicenseValidator(String licensePath, String publicKeyPath) {
        this.licensePath = licensePath;
        this.publicKeyPath = publicKeyPath;
        this.timeFormat.applyPattern("yyyy-MM-dd HH:mm:ss");
        this.validateSystemClock();
        this.loadLicenses();
        this.loadPersistData();
        this.loadPublicKey();
        if (!hostLoaded) {
            TLicenseValidator.loadHostType(this.licensePath);
        }
    }

    public TLicenseValidator(String licensePath) {
        this(licensePath, null);
    }

    private void loadLicenses() {
        if (this.licensePath == null || this.licensePath.isEmpty()) {
            logger.error("License folder not provided");
            return;
        }
        File dir = new File(this.licensePath);
        FilenameFilter filter = new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith("lic");
            }
        };
        if (!dir.exists()) {
            logger.error("License folder does not exist: " + this.licensePath);
            return;
        }
        File[] files = dir.listFiles(filter);
        if (files == null) {
            logger.error("No license files found in path: " + this.licensePath);
            return;
        }
        File[] fileArray = files;
        int n = fileArray.length;
        for (int i = 0; i < n; ++i) {
            TLicense license = new TLicense();
            File file = fileArray[i];
            if (license.load(file.getAbsolutePath())) {
                TLicenseHolder.TLoadLicErr ret = this.loadLicense(license);
                if (ret == TLicenseHolder.TLoadLicErr.errOk || ret == TLicenseHolder.TLoadLicErr.errExpired) {
                    this.licenses.add(new TLicenseHolder(file.getName(), ret, license));
                } else {
                    this.licenses.add(new TLicenseHolder(file.getName(), ret));
                }
                logger.info("License file found in " + file.getName() + " [" + (ret == TLicenseHolder.TLoadLicErr.errOk || ret == TLicenseHolder.TLoadLicErr.errExpired ? "" : "NOT ") + "LOADED]");
                continue;
            }
            this.licenses.add(new TLicenseHolder(file.getName()));
        }
    }

    private TLicenseHolder.TLoadLicErr loadLicense(TLicense aLic) {
        if (aLic.getFLicType() == TLicense.TLicType.LIC_INVALID) {
            return TLicenseHolder.TLoadLicErr.errInvalid;
        }
        if (!aLic.checkExpiredDate().booleanValue()) {
            logger.error("License expired");
            return TLicenseHolder.TLoadLicErr.errExpired;
        }
        if (this.licenseByProductId(aLic.getFUuid()) != null) {
            logger.error("License " + aLic.getFLicId() + " not loaded: Already exists a product " + aLic.getFUuid() + " license");
            return TLicenseHolder.TLoadLicErr.errAExists;
        }
        return TLicenseHolder.TLoadLicErr.errOk;
    }

    private TLicense licenseByProductId(UUID fUuid) {
        for (TLicenseHolder lic : this.licenses) {
            if (lic.getfStatus() != TLicenseHolder.TLoadLicErr.errOk && lic.getfStatus() != TLicenseHolder.TLoadLicErr.errExpired || !lic.getfLicense().getFUuid().equals(fUuid)) continue;
            return lic.getfLicense();
        }
        return null;
    }

    private TLicenseHolder licenseHolderByProductId(UUID fUuid) {
        for (TLicenseHolder lic : this.licenses) {
            if (lic.getfLicense() == null || !lic.getfLicense().getFUuid().equals(fUuid)) continue;
            return lic;
        }
        return null;
    }

    private void loadPublicKey() {
        try {
            RSAPublicKeySpec keySpec = null;
            if (this.publicKeyPath != null) {
                keySpec = this.readPublicKey();
            } else {
                BigInteger e = new BigInteger(Constants.PK_EXPONENT, 10);
                BigInteger n = new BigInteger(Constants.PK_MODULUS, 10);
                keySpec = new RSAPublicKeySpec(n, e);
            }
            KeyFactory keyFactory = KeyFactory.getInstance("RSA", "SunJSSE");
            this.publicKey = keyFactory.generatePublic(keySpec);
            logger.info("Loaded public key");
        }
        catch (Exception ex) {
            logger.error(ex.getLocalizedMessage());
            this.publicKey = null;
        }
    }

    private RSAPublicKeySpec readPublicKey() throws LicenseException {
        try {
            File file = new File(this.publicKeyPath);
            FileInputStream keyFile = new FileInputStream(file);
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            while (((InputStream)keyFile).available() != 0) {
                bos.write(((InputStream)keyFile).read());
            }
            ByteBuffer bb = ByteBuffer.wrap(bos.toByteArray());
            bb.order(ByteOrder.BIG_ENDIAN);
            bb.getInt();
            int bLen = bb.getInt();
            byte[] b = new byte[bLen];
            bb.get(b);
            BigInteger e = new BigInteger(1, b);
            bLen = bb.getInt();
            b = new byte[bLen];
            bb.get(b);
            BigInteger n = new BigInteger(1, b);
            bos.close();
            ((InputStream)keyFile).close();
            return new RSAPublicKeySpec(n, e);
        }
        catch (Exception ex) {
            logger.error("Public key not loaded: " + ex.getLocalizedMessage());
            throw new LicenseException("Invalid license file");
        }
    }

    public Boolean validateProduct(UUID prodUuid, String prodVer, String prodName, Date prodDate, boolean saveError) {
        Constants.TLicenseError licErr = Constants.TLicenseError.E_LIC_FATAL;
        if (this.licenses.size() == 0) {
            return false;
        }
        this.validateSystemClock();
        if (this.fLastErr != null && this.fLastErr != Constants.TLicenseError.E_LIC_OK) {
            if (saveError) {
                this.appendProductError(new TProductError(prodUuid, prodVer, prodName, licErr));
            }
            return false;
        }
        if (this.publicKey == null) {
            licErr = Constants.TLicenseError.E_LIC_KEYNOTFOUND;
        } else {
            licErr = this.validateProduct(prodUuid, prodVer, prodDate, this.publicKey, saveError);
            if (licErr != Constants.TLicenseError.E_LIC_OK && licErr != Constants.TLicenseError.E_LIC_UNKNOWN && licErr != Constants.TLicenseError.E_LIC_NOTFOUND && saveError) {
                this.appendProductError(new TProductError(prodUuid, prodVer, prodName, licErr));
            }
        }
        this.savePersistData();
        return licErr == Constants.TLicenseError.E_LIC_OK;
    }

    private Constants.TLicenseError validateProduct(UUID prodUuid, String prodVer, Date prodDate, PublicKey pubKey, boolean saveError) {
        TLicenseHolder holder = this.licenseHolderByProductId(prodUuid);
        if (holder == null) {
            return Constants.TLicenseError.E_LIC_NOTFOUND;
        }
        TLicense lic = holder.getfLicense();
        if (lic == null) {
            holder.setfLicStatus(Constants.TLicenseError.E_LIC_NOTFOUND);
            return Constants.TLicenseError.E_LIC_NOTFOUND;
        }
        if (lic.getFLicType() == TLicense.TLicType.LIC_INVALID) {
            holder.setfLicStatus(Constants.TLicenseError.E_LIC_INVALID);
            return Constants.TLicenseError.E_LIC_INVALID;
        }
        if (!lic.verify(pubKey)) {
            holder.setfLicStatus(Constants.TLicenseError.E_LIC_INVALID_SIGN);
            return Constants.TLicenseError.E_LIC_INVALID_SIGN;
        }
        if (lic.getFLicType() == TLicense.TLicType.LIC_DEBUG) {
            holder.setfLicStatus(Constants.TLicenseError.E_LIC_OK);
            return Constants.TLicenseError.E_LIC_OK;
        }
        if (!lic.checkProductVersion(prodVer, prodDate).booleanValue()) {
            holder.setfLicStatus(Constants.TLicenseError.E_LIC_INVALID_VERSION);
            return Constants.TLicenseError.E_LIC_INVALID_VERSION;
        }
        if (!lic.checkExpiredDate(prodDate).booleanValue()) {
            holder.setfLicStatus(Constants.TLicenseError.E_LIC_EXPIRED);
            return Constants.TLicenseError.E_LIC_EXPIRED;
        }
        if (lic.getFLicType() == TLicense.TLicType.LIC_SINGLE || lic.getFLicType() == TLicense.TLicType.LIC_DEMO) {
            if (!lic.checkHost().booleanValue()) {
                holder.setfLicStatus(Constants.TLicenseError.E_LIC_INVALID_HOST);
                return Constants.TLicenseError.E_LIC_INVALID_HOST;
            }
        } else {
            if (holder.isServerBased().booleanValue()) {
                return Constants.TLicenseError.E_LIC_OK;
            }
            holder.setfLicStatus(Constants.TLicenseError.E_LIC_FATAL);
            return Constants.TLicenseError.E_LIC_FATAL;
        }
        holder.setfLicStatus(Constants.TLicenseError.E_LIC_OK);
        return Constants.TLicenseError.E_LIC_OK;
    }

    public Boolean validateProductUserCount(UUID prodUuid, int userCount) {
        Constants.TLicenseError licErr = Constants.TLicenseError.E_LIC_OK;
        if (this.licenses.size() == 0) {
            return false;
        }
        TLicenseHolder holder = this.licenseHolderByProductId(prodUuid);
        if (holder == null) {
            licErr = Constants.TLicenseError.E_LIC_NOTFOUND;
            return false;
        }
        TLicense lic = holder.getfLicense();
        if (lic == null) {
            holder.setfLicStatus(Constants.TLicenseError.E_LIC_NOTFOUND);
            licErr = Constants.TLicenseError.E_LIC_NOTFOUND;
            return false;
        }
        if (lic.getFCount() < userCount) {
            holder.setfLicStatus(Constants.TLicenseError.E_LIC_NOTAVAIL);
            licErr = Constants.TLicenseError.E_LIC_NOTAVAIL;
        }
        return licErr == Constants.TLicenseError.E_LIC_OK;
    }

    public boolean hasLicenseExpirationDate(UUID prodUuid) {
        if (this.licenses.size() == 0) {
            return false;
        }
        TLicenseHolder holder = this.licenseHolderByProductId(prodUuid);
        if (holder == null) {
            return false;
        }
        TLicense lic = holder.getfLicense();
        if (lic == null) {
            holder.setfLicStatus(Constants.TLicenseError.E_LIC_NOTFOUND);
            return false;
        }
        return lic.getFExpireDate() != null;
    }

    public int getLicenseDaysLeft(UUID prodUuid) {
        if (this.licenses.size() == 0) {
            return -1;
        }
        TLicenseHolder holder = this.licenseHolderByProductId(prodUuid);
        if (holder == null) {
            return -1;
        }
        TLicense lic = holder.getfLicense();
        if (lic == null) {
            holder.setfLicStatus(Constants.TLicenseError.E_LIC_NOTFOUND);
            return -1;
        }
        if (!lic.checkExpiredDate().booleanValue()) {
            holder.setfLicStatus(Constants.TLicenseError.E_LIC_EXPIRED);
            return 0;
        }
        if (lic.getFExpireDate() != null) {
            Long timeDiff = lic.getFExpireDate().getTime() - System.currentTimeMillis();
            return (int)Math.ceil((float)timeDiff.longValue() / 8.64E7f);
        }
        return Integer.MAX_VALUE;
    }

    private Constants.TLicenseError validateSystemClock() {
        this.loadSystemRunInfo();
        Date now = new Date();
        if (this.fLastRun.before(this.f1stRun) || this.fLastRun.after(now)) {
            this.fLastErr = Constants.TLicenseError.E_LIC_TIMEFAKED;
            return this.fLastErr;
        }
        this.fLastRun = now;
        this.fLastErr = Constants.TLicenseError.E_LIC_OK;
        this.saveSystemRunInfo();
        return this.fLastErr;
    }

    private void loadSystemRunInfo() {
        if (!this.loadSystemRunInfo(this.fTempFile)) {
            this.f1stRun = new Date();
            this.fLastRun = new Date();
        }
    }

    private boolean loadSystemRunInfo(String tempFileName) {
        try {
            File tempFile = new File(tempFileName);
            FileInputStream fis = new FileInputStream(tempFile);
            ByteArrayOutputStream buffer = new ByteArrayOutputStream((int)tempFile.length());
            while (fis.available() != 0) {
                buffer.write(fis.read());
            }
            fis.close();
            String data = new String(Base64.decodeBase64((String)new String(buffer.toByteArray())));
            String[] dataArray = data.split(",");
            this.f1stRun = this.timeFormat.parse(dataArray[0]);
            this.fLastRun = this.timeFormat.parse(dataArray[1]);
            return true;
        }
        catch (Exception ex) {
            logger.error(ex.getLocalizedMessage());
            this.fLastErr = Constants.TLicenseError.E_LIC_TEMP_RIGHTS;
            return false;
        }
    }

    private void saveSystemRunInfo() {
        String data = this.timeFormat.format(this.f1stRun) + "," + this.timeFormat.format(new Date());
        try {
            File tempFile = new File(this.fTempFile);
            FileOutputStream fos = new FileOutputStream(tempFile);
            fos.write(Base64.encodeBase64String((byte[])data.getBytes()).getBytes());
            fos.close();
        }
        catch (Exception ex) {
            logger.error(ex.getLocalizedMessage());
            this.fLastErr = Constants.TLicenseError.E_LIC_TEMP_RIGHTS;
        }
    }

    private void savePersistData() {
    }

    private void loadPersistData() {
    }

    private void appendProductError(TProductError aError) {
        if (!aError.isValid().booleanValue()) {
            return;
        }
        for (TProductError e : this.fErrorsList) {
            if (!e.getfProdId().equals(aError.getfProdId())) continue;
            this.fErrorsList.add(this.fErrorsList.indexOf(e), aError);
            this.fErrorsList.remove(e);
            return;
        }
        this.fErrorsList.add(aError);
    }

    public static String errorString(Constants.TLicenseError errorCode) {
        String errorTxt = "";
        switch (errorCode) {
            case E_LIC_OK: {
                break;
            }
            case E_LIC_FATAL: {
                errorTxt = "License checking fatal error. Please, contact support@mestrelab.com";
                break;
            }
            case E_LIC_EXPIRED: {
                errorTxt = "The license period has expired";
                break;
            }
            case E_LIC_ALPHA_EXPIRED: {
                errorTxt = "The alpha version has expired";
                break;
            }
            case E_LIC_TIMEFAKED: 
            case E_LIC_SRV_BADDATE: {
                errorTxt = "It was detected that the system time has been changed unexpectedly or is invalid. Please, contact support@mestrelab.com";
                break;
            }
            case E_LIC_SRV_BADSIGN: 
            case E_LIC_INVALID_SIGN: {
                errorTxt = "The license cannot be verified";
                break;
            }
            case E_LIC_INVALID_VERSION: {
                errorTxt = "You are not entitled to run this version because your Update&Support has expired. Please, contact sales@mestrelab.com to renew your U&S or downgrade to an older version";
                break;
            }
            case E_LIC_INVALID_HOST: 
            case E_LIC_INVALID: {
                errorTxt = "The license is invalid";
                break;
            }
            case E_LIC_KEYNOTFOUND: {
                errorTxt = "Unable to load a valid key";
                break;
            }
            case E_LIC_NOTFOUND: {
                errorTxt = "A valid license file was not found for this product";
                break;
            }
            case E_LIC_SRV_CONNECT: {
                errorTxt = "Failed to connect to license server. Contact your system administrator or support@mestrelab.com";
                break;
            }
            case E_LIC_SRV_PROTO: 
            case E_LIC_SRV_BADREQUEST: 
            case E_LIC_SRV_BADRESPONSE: {
                errorTxt = "License server protocol error. Please, try again";
                break;
            }
            case E_LIC_NOTAVAIL: {
                errorTxt = "Reached the user limit";
                break;
            }
            case E_LIC_TEMP_RIGHTS: {
                errorTxt = "You don't have the needed rights. Please, contact support@mestrelab.com";
                break;
            }
            case E_LIC_SRV_SESSIONNOTFOUND: {
                errorTxt = "License session not found. Please try again to validate your products";
                break;
            }
            case E_LIC_SRV_DENY: {
                errorTxt = "License server forbids you to get a license. Contact your system administrator";
                break;
            }
            default: {
                errorTxt = "An error happened while checking licenses. Please, contact support@mestrelab.com";
            }
        }
        if (errorTxt.length() > 0) {
            logger.error(errorTxt);
        }
        return errorTxt;
    }

    public String errorString(UUID aProdUuid) {
        if (this.licenses.size() == 0) {
            return TLicenseValidator.errorString(Constants.TLicenseError.E_LIC_FATAL);
        }
        if (this.publicKey == null) {
            return TLicenseValidator.errorString(Constants.TLicenseError.E_LIC_KEYNOTFOUND);
        }
        if (this.fLastErr != null && this.fLastErr != Constants.TLicenseError.E_LIC_OK) {
            return TLicenseValidator.errorString(this.fLastErr);
        }
        TLicense lic = this.licenseByProductId(aProdUuid);
        if (lic == null) {
            return TLicenseValidator.errorString(Constants.TLicenseError.E_LIC_NOTFOUND);
        }
        TLicenseHolder licInfo = this.licenseHolderByProductId(aProdUuid);
        if (licInfo == null) {
            return TLicenseValidator.errorString(Constants.TLicenseError.E_LIC_NOTFOUND);
        }
        return TLicenseValidator.errorString(licInfo.getfLicStatus());
    }

    public String getHostId() {
        return TLicense.getHostId();
    }

    public static String getHostId(String licensePath) {
        TLicenseValidator.loadHostType(licensePath);
        hostLoaded = true;
        return TLicense.getHostId();
    }

    private static void loadHostType(String licensePath) {
        Properties prop = new Properties();
        try {
            FileInputStream input = new FileInputStream(licensePath + System.getProperty("file.separator") + "license.properties");
            prop.load(input);
            String hostType = prop.getProperty("HostType");
            if (hostType.equals("1")) {
                Constants.gHostIdType = Constants.THostIdType.AWS;
            } else if (!hostType.equals("0")) {
                logger.info("Host Type incorrect value. Assumed default host type");
            }
            ((InputStream)input).close();
        }
        catch (IOException ex) {
            logger.error(ex.getLocalizedMessage());
            logger.info("Host Type not loaded. Assumed default host type");
        }
        logger.info("Host Type value: " + Constants.gHostIdType.getValue());
    }
}

